const express = require("express");
const fs = require("fs");
const path = require("path");
const crypto = require("crypto");
const session = require("express-session");
const chalk = require("chalk");
const cors = require("cors");
const axios = require("axios");
const multer = require("multer");
const { formidable } = require("formidable");
const cron = require("node-cron"); // <-- tambahkan

const app = express();
const PORT = process.env.PORT || 3000;

// ==================== GLOBAL VARIABLES ====================
global.pwadmin = "123456"; // password admin (bisa diubah)
global.getBuffer = async (url, options = {}) => {
  try {
    const res = await axios({ method: 'get', url, headers: { 'DNT': 1, 'Upgrade-Insecure-Request': 1 }, ...options, responseType: 'arraybuffer' });
    return res.data;
  } catch (err) { return err; }
};
global.fetchJson = async (url, options = {}) => {
  try {
    const res = await axios({ method: 'GET', url, headers: { 'User-Agent': 'Mozilla/5.0' }, ...options });
    return res.data;
  } catch (err) { return err; }
};

// ==================== DATABASE PATH & HELPERS ====================
const dbPath = path.join(__dirname, "database");
function read(file) {
  return JSON.parse(fs.readFileSync(path.join(dbPath, file)));
}
function write(file, data) {
  fs.writeFileSync(path.join(dbPath, file), JSON.stringify(data, null, 2));
}

// ==================== SECURITY FUNCTIONS ====================
function hashPassword(password, salt) {
  return crypto.pbkdf2Sync(password, salt, 10000, 64, "sha512").toString("hex");
}
function generateToken() {
  return crypto.randomBytes(32).toString("hex");
}
function generateApiKey() {
  return crypto.randomBytes(3).toString("hex");
}

// ==================== MIDDLEWARE GLOBAL ====================
app.enable("trust proxy");
app.set("json spaces", 2);
app.use(express.json());
app.use(express.urlencoded({ extended: true }));
app.use(cors());
app.use(express.static(path.join(__dirname, 'public')));  // untuk halaman frontend

// ==================== SESSION ====================
app.use(session({
  secret: "secret_key_masharunapi",
  resave: false,
  saveUninitialized: false,
  cookie: {
    maxAge: 60 * 60 * 1000, // 1 jam
    httpOnly: true,
    sameSite: "lax",
    secure: false
  }
}));

// ==================== AUTH MIDDLEWARE ====================
function userAuth(req, res, next) {
  if (!req.session.user) {
    return res.status(401).json({ success: false, message: "Unauthorized" });
  }
  next();
}
function adminAuth(req, res, next) {
  if (!req.session.admin) {
    return res.status(401).json({ success: false, message: "Unauthorized" });
  }
  next();
}

// ==================== API KEY MIDDLEWARE ====================
const apiKeyMiddleware = async (req, res, next) => {
  const apikey = req.query.apikey || req.headers['x-api-key'] || req.headers['authorization']?.replace('Bearer ', '');
  if (!apikey) {
    return res.status(401).json({ error: 'API key diperlukan' });
  }

  try {
    const users = read("users.json");
    const user = users.find(u => u.apiKey === apikey);
    if (!user) {
      return res.json({ status: false, result: 'APIkey tidak valid.' });
    }
    if (user.suspended) {
      return res.json({ status: false, result: 'Akun telah di Suspend.' });
    }
    if (user.expiresAt && new Date(user.expiresAt) < new Date()) {
      return res.json({ status: false, result: 'APIkey sudah Kedaluwarsa atau Expired' });
    }
    // Cek limit (unlimited = 999999)
    const limit = user.limit === 999999 ? Infinity : user.limit;
    if (user.usage >= limit) {
      return res.json({ status: false, result: 'Limit telah habis! limit direset jam 00:00, Upgrade Role! untuk mendapatkan banyak limit.' });
    }
    req.user = user;
    next();
  } catch (err) {
    console.error('Auth error:', err);
    res.status(500).json({ error: 'Kesalahan server' });
  }
};

// Fungsi untuk increment usage setelah response sukses
const incrementUsage = (user) => {
  try {
    const users = read("users.json");
    const userIndex = users.findIndex(u => u.id === user.id);
    if (userIndex !== -1) {
      users[userIndex].usage = (users[userIndex].usage || 0) + 1;
      write("users.json", users);
    }
  } catch (err) {
    console.error('Error updating usage:', err);
  }
};

// ==================== LOAD API ENDPOINTS OTOMATIS DARI FOLDER ./api ====================
let totalRoutes = 0;
let rawEndpoints = {};

const apiFolder = path.join(__dirname, './api');
const upload = multer();

// Fungsi helper untuk konversi parameter
function convertParametersForFrontend(parameters) {
  if (!parameters) return {};
  const converted = {};
  for (const [paramName, paramConfig] of Object.entries(parameters)) {
    converted[paramName] = {
      type: paramConfig.type || "string",
      ...(paramConfig.required !== undefined && { required: paramConfig.required }),
      ...(paramConfig.example && { example: paramConfig.example }),
      ...(paramConfig.value && { value: paramConfig.value }),
      ...(paramConfig.selection && { selection: paramConfig.selection })
    };
  }
  return converted;
}

const register = (ep, file) => {
  if (ep && ep.name && (ep.desc || ep.description) && ep.category && ep.path && typeof ep.run === "function") {
    const cleanPath = ep.path.split("?")[0];
    const method = ep.method ? ep.method.toLowerCase() : 'get';

    // Tentukan apakah endpoint memerlukan API key (default true)
    const requiresAuth = ep.authRequired !== false;

    // Buat handler yang membungkus run asli dengan increment usage setelah response
    const wrappedHandler = async (req, res) => {
      // Simpan user dari middleware (jika ada)
      const user = req.user;

      // Pasang listener untuk menambah usage setelah response sukses
      res.on('finish', () => {
        if (user && res.statusCode >= 200 && res.statusCode < 300) {
          incrementUsage(user);
        }
      });

      try {
        await ep.run(req, res);
      } catch (err) {
        // Biarkan express menangani error
        throw err;
      }
    };

    // Daftarkan route dengan middleware yang sesuai
    if (method === 'post') {
      if (requiresAuth) {
        app.post(cleanPath, upload.any(), apiKeyMiddleware, wrappedHandler);
      } else {
        app.post(cleanPath, upload.any(), wrappedHandler);
      }
    } else {
      if (requiresAuth) {
        app.get(cleanPath, apiKeyMiddleware, wrappedHandler);
      } else {
        app.get(cleanPath, wrappedHandler);
      }
    }

    // Catat endpoint untuk frontend
    if (!rawEndpoints[ep.category]) rawEndpoints[ep.category] = [];
    const endpointData = {
      name: ep.name,
      description: ep?.description || ep?.desc || null,
      path: ep.path,
      method: ep.method || 'GET',
      parameters: convertParametersForFrontend(ep.parameters),
      ...(ep.innerDesc ? { innerDesc: ep.innerDesc } : {}),
      ...(ep.body ? { body: ep.body } : {})
    };
    rawEndpoints[ep.category].push(endpointData);
    totalRoutes++;
    console.log(chalk.bgHex('#FFFF99').hex('#333').bold(` Loaded Route: ${file} → ${ep.name} (${method.toUpperCase()}) `));
    if (ep.parameters) {
      console.log(chalk.hex('#FFA500')(`  Parameters: ${Object.keys(ep.parameters).join(', ')}`));
    }
  }
};

// Baca semua file di folder api
if (fs.existsSync(apiFolder)) {
  fs.readdirSync(apiFolder).forEach((file) => {
    const filePath = path.join(apiFolder, file);
    if (path.extname(file) === '.js') {
      try {
        delete require.cache[require.resolve(filePath)];
        const routeModule = require(filePath);
        if (Array.isArray(routeModule)) {
          routeModule.forEach(ep => register(ep, file));
        } else if (routeModule.endpoint) {
          register(routeModule.endpoint, file);
        } else if (typeof routeModule === "function") {
          routeModule(app);
        } else {
          register(routeModule, file);
        }
      } catch (err) {
        console.error(chalk.red(`Error loading ${file}:`), err.message);
      }
    }
  });
  console.log(chalk.bgHex('#90EE90').hex('#333').bold(' Load Complete! ✓ '));
  console.log(chalk.bgHex('#90EE90').hex('#333').bold(` Total Routes Loaded: ${totalRoutes} `));
} else {
  console.log(chalk.yellow('Folder ./api tidak ditemukan, lewati loading endpoint.'));
}

// ==================== ENDPOINT KHUSUS DARI API AUTO (settings, docs, category) ====================
app.get('/settings', (req, res) => {
  // Gabungkan dengan settings dari database jika ada
  let dbSettings = {};
  try {
    dbSettings = read("settings.json");
  } catch (e) {
    dbSettings = { title: "Sky Rest API", description: "Powerful REST API Service" };
  }

  const endpoints = {
    categories: Object.keys(rawEndpoints)
      .sort((a, b) => a.localeCompare(b))
      .map(category => ({
        name: category,
        items: rawEndpoints[category]
          .sort((a, b) => a.name.localeCompare(b.name))
          .map(endpoint => ({
            name: endpoint.name,
            method: endpoint.method || 'GET',
            path: endpoint.path,
            description: endpoint.description || endpoint.desc,
            parameters: endpoint.parameters || {}
          }))
      }))
  };

  const fullSettings = {
    ...dbSettings,
    categories: endpoints.categories,
    metadata: {
      totalEndpoints: totalRoutes,
      totalCategories: endpoints.categories.length,
      lastUpdated: new Date().toISOString()
    }
  };
  res.json(fullSettings);
});

app.get('/api', (req, res) => {
  res.sendFile(path.join(__dirname, 'public/docs.html'));
});

// Route per kategori
Object.keys(rawEndpoints).forEach(category => {
  const slug = '/' + category.toLowerCase().replace(/\s+/g, '-');
  app.get(slug, (req, res) => {
    const items = rawEndpoints[category].map(endpoint => ({
      name: endpoint.name,
      method: endpoint.method || 'GET',
      path: endpoint.path,
      description: endpoint.description || endpoint.desc,
      parameters: endpoint.parameters || {}
    }));
    res.json({ success: true, category, items });
  });
  console.log(chalk.cyan(`  Category route: ${slug} → ${category}`));
});

// ==================== ROUTE UNTUK HALAMAN FRONTEND (USER) ====================
app.get("/", (req, res) => res.redirect("/login"));
app.get("/login", (req, res) => res.sendFile(path.join(__dirname, "public/login.html")));
app.get("/register", (req, res) => res.sendFile(path.join(__dirname, "public/register.html")));
app.get("/dashboard", (req, res) => {
  if (!req.session.user) return res.redirect("/login");
  res.sendFile(path.join(__dirname, "public/dashboard.html"));
});
app.get("/profile", (req, res) => {
  if (!req.session.user) return res.redirect("/login");
  res.sendFile(path.join(__dirname, "public/profile.html"));
});

// ==================== API UNTUK USER ====================
app.post("/api/register", (req, res) => {
  const { name, email, password, package: packageName } = req.body;
  if (!name || !email || !password) {
    return res.json({ success: false, message: "All fields required" });
  }

  const users = read("users.json");
  const packages = read("packages.json");

  if (users.find(u => u.email === email)) {
    return res.json({ success: false, message: "Email already used" });
  }
  let selectedPackage = packages.find(p => p.name === packageName) || packages.find(p => p.name === "Free");
  if (!selectedPackage) {
    return res.json({ success: false, message: "Package not found" });
  }

  const salt = crypto.randomBytes(16).toString("hex");
  const hash = hashPassword(password, salt);
  const expiresAt = new Date();
  expiresAt.setMonth(expiresAt.getMonth() + (selectedPackage.duration || 1));

  let limitValue = selectedPackage.limit;
  if (limitValue === "unlimited") {
    limitValue = 999999;
  } else {
    limitValue = parseInt(limitValue);
    if (isNaN(limitValue)) limitValue = 0;
  }

  const newUser = {
    id: crypto.randomBytes(8).toString("hex"),
    name,
    email,
    password_hash: hash,
    password_salt: salt,
    role: "user",
    apiKey: generateApiKey(),
    package: selectedPackage.name,
    limit: limitValue,
    usage: 0,
    suspended: false,
    createdAt: new Date().toISOString(),
    expiresAt: expiresAt.toISOString()
  };

  users.push(newUser);
  write("users.json", users);
  res.json({ success: true, message: "Register success" });
});

app.post("/api/login", (req, res) => {
  const { email, password } = req.body;
  if (!email || !password) {
    return res.json({ success: false, message: "Email & password required" });
  }
  try {
    const users = read("users.json");
    const user = users.find(u => u.email === email);
    if (!user) {
      return res.json({ success: false, message: "User not found" });
    }
    const verify = hashPassword(password, user.password_salt);
    if (verify !== user.password_hash) {
      return res.json({ success: false, message: "Wrong password" });
    }
    req.session.user = {
      id: user.id,
      name: user.name,
      role: user.role
    };
    req.session.save(err => {
      if (err) {
        console.error("Session save error:", err);
        return res.json({ success: false, message: "Session error" });
      }
      res.json({ success: true, user: { id: user.id, name: user.name, role: user.role } });
    });
  } catch (error) {
    console.error("Login error:", error);
    res.json({ success: false, message: "Server error" });
  }
});

app.get("/api/profile", userAuth, (req, res) => {
  try {
    const users = read("users.json");
    const user = users.find(u => u.id === req.session.user.id);
    if (!user) {
      req.session.destroy();
      return res.json({ success: false, message: "User not found" });
    }
    res.json({
      success: true,
      name: user.name,
      email: user.email,
      package: user.package,
      limit: user.limit,
      usage: user.usage,
      apiKey: user.apiKey,
      suspended: user.suspended,
      expiresAt: user.expiresAt
    });
  } catch (error) {
    console.error("Profile error:", error);
    res.json({ success: false, message: "Server error" });
  }
});

app.put("/api/profile", userAuth, (req, res) => {
  const { name, email, newPassword } = req.body;
  try {
    const users = read("users.json");
    const userIndex = users.findIndex(u => u.id === req.session.user.id);
    if (userIndex === -1) {
      return res.json({ success: false, message: "User tidak ditemukan" });
    }
    if (email && email !== users[userIndex].email) {
      const emailExists = users.find(u => u.email === email && u.id !== req.session.user.id);
      if (emailExists) {
        return res.json({ success: false, message: "Email sudah digunakan oleh akun lain" });
      }
      users[userIndex].email = email;
    }
    if (name) {
      users[userIndex].name = name;
      req.session.user.name = name;
    }
    if (newPassword && newPassword.trim() !== "") {
      const salt = crypto.randomBytes(16).toString("hex");
      users[userIndex].password_salt = salt;
      users[userIndex].password_hash = hashPassword(newPassword, salt);
    }
    write("users.json", users);
    res.json({ success: true, message: "Profil berhasil diperbarui" });
  } catch (error) {
    console.error("Update profile error:", error);
    res.status(500).json({ success: false, message: "Terjadi kesalahan server" });
  }
});

app.post("/api/profile/regenerate-key", userAuth, (req, res) => {
  try {
    const users = read("users.json");
    const index = users.findIndex(u => u.id === req.session.user.id);
    if (index === -1) return res.json({ success: false });
    const newKey = generateApiKey()
    users[index].apiKey = newKey;
    write("users.json", users);
    res.json({ success: true, apiKey: newKey, message: "API Key berhasil diperbarui" });
  } catch (e) {
    res.status(500).json({ success: false, message: "Gagal membuat API Key baru" });
  }
});

app.get("/api/logout", (req, res) => {
  req.session.destroy(err => {
    if (err) return res.json({ success: false });
    res.json({ success: true });
  });
});

// ==================== ANNOUNCEMENTS API ====================
const announcementsPath = path.join(dbPath, "announcements.json");
function readAnnouncements() {
  try {
    return JSON.parse(fs.readFileSync(announcementsPath));
  } catch (error) {
    return [];
  }
}
function writeAnnouncements(data) {
  fs.writeFileSync(announcementsPath, JSON.stringify(data, null, 2));
}

app.get("/api/admin/announcements", (req, res) => {
  try {
    res.json(readAnnouncements());
  } catch (error) {
    res.json([]);
  }
});

app.post("/api/admin/announcements", adminAuth, (req, res) => {
  const { title, content, active } = req.body;
  if (!title || !content) {
    return res.json({ success: false, message: "Title and content required" });
  }
  try {
    const announcements = readAnnouncements();
    const newAnn = {
      id: crypto.randomBytes(8).toString("hex"),
      title,
      content,
      active: active || false,
      createdAt: new Date().toISOString(),
      updatedAt: new Date().toISOString()
    };
    announcements.push(newAnn);
    writeAnnouncements(announcements);
    res.json({ success: true, announcement: newAnn });
  } catch (error) {
    res.json({ success: false, message: "Server error" });
  }
});

app.put("/api/admin/announcements/:id", adminAuth, (req, res) => {
  const { id } = req.params;
  const { title, content, active } = req.body;
  try {
    const announcements = readAnnouncements();
    const index = announcements.findIndex(a => a.id === id);
    if (index === -1) {
      return res.json({ success: false, message: "Announcement not found" });
    }
    announcements[index] = {
      ...announcements[index],
      title: title || announcements[index].title,
      content: content || announcements[index].content,
      active: active !== undefined ? active : announcements[index].active,
      updatedAt: new Date().toISOString()
    };
    writeAnnouncements(announcements);
    res.json({ success: true, announcement: announcements[index] });
  } catch (error) {
    res.json({ success: false, message: "Server error" });
  }
});

app.delete("/api/admin/announcements/:id", adminAuth, (req, res) => {
  const { id } = req.params;
  try {
    let announcements = readAnnouncements();
    const filtered = announcements.filter(a => a.id !== id);
    if (filtered.length === announcements.length) {
      return res.json({ success: false, message: "Announcement not found" });
    }
    writeAnnouncements(filtered);
    res.json({ success: true });
  } catch (error) {
    res.json({ success: false, message: "Server error" });
  }
});

app.get("/api/announcements", (req, res) => {
  try {
    const announcements = readAnnouncements();
    const active = announcements
      .filter(a => a.active)
      .sort((a, b) => new Date(b.createdAt) - new Date(a.createdAt));
    res.json(active);
  } catch (error) {
    res.json([]);
  }
});

// ==================== ADMIN PAGES ====================
app.get("/admin/login", (req, res) => res.sendFile(path.join(__dirname, "public/admin-login.html")));
app.get("/admin", (req, res) => {
  if (!req.session.admin) return res.redirect("/admin/login");
  res.sendFile(path.join(__dirname, "public/admin-dashboard.html"));
});
app.get("/admin/manage-user", (req, res) => {
  if (!req.session.admin) return res.redirect("/admin/login");
  res.sendFile(path.join(__dirname, "public/admin-manage-user.html"));
});
app.get("/admin/detail-user", (req, res) => {
  if (!req.session.admin) return res.redirect("/admin/login");
  res.sendFile(path.join(__dirname, "public/admin-detail-user.html"));
});
app.get("/admin/settings", (req, res) => {
  if (!req.session.admin) return res.redirect("/admin/login");
  res.sendFile(path.join(__dirname, "public/admin-settings.html"));
});
app.get("/admin/packages", (req, res) => {
  if (!req.session.admin) return res.redirect("/admin/login");
  res.sendFile(path.join(__dirname, "public/admin-packages.html"));
});

// ==================== ADMIN API ====================
app.post("/api/admin/login", (req, res) => {
  const { password } = req.body;
  if (!password) return res.json({ success: false, message: "Password required" });
  if (password !== global.pwadmin) return res.json({ success: false, message: "Wrong password" });

  req.session.admin = { token: generateToken() };
  req.session.save(err => {
    if (err) console.error("Session save error:", err);
    res.json({ success: true, message: "Admin logged in" });
  });
});

app.get("/api/admin/logout", (req, res) => {
  req.session.destroy(err => {
    if (err) return res.json({ success: false });
    res.json({ success: true });
  });
});

app.get("/api/admin/stats", adminAuth, (req, res) => {
  try {
    const users = read("users.json");
    const totalUsers = users.length;
    const suspended = users.filter(u => u.suspended).length;
    const active = totalUsers - suspended;
    const totalUsage = users.reduce((a, b) => a + (b.usage || 0), 0);
    res.json({ totalUsers, active, suspended, totalUsage });
  } catch (error) {
    res.json({ success: false, message: "Server error" });
  }
});

app.post("/api/admin/update-user-full", adminAuth, (req, res) => {
  const { userId, name, email, apiKey, package, limit, usage, password } = req.body;
  try {
    const users = read("users.json");
    const userIndex = users.findIndex(u => u.id === userId);
    if (userIndex === -1) return res.json({ success: false, message: "User tidak ditemukan" });

    users[userIndex].name = name || users[userIndex].name;
    users[userIndex].email = email || users[userIndex].email;
    users[userIndex].apiKey = apiKey || users[userIndex].apiKey;
    users[userIndex].package = package || users[userIndex].package;
    users[userIndex].limit = parseInt(limit);
    users[userIndex].usage = parseInt(usage);

    if (password && password.trim() !== "") {
      const salt = crypto.randomBytes(16).toString("hex");
      users[userIndex].password_salt = salt;
      users[userIndex].password_hash = hashPassword(password, salt);
    }

    write("users.json", users);
    res.json({ success: true, message: "User berhasil diperbarui" });
  } catch (error) {
    res.json({ success: false, message: "Server error" });
  }
});

app.get("/api/admin/users", adminAuth, (req, res) => {
  try {
    res.json(read("users.json"));
  } catch (error) {
    res.json([]);
  }
});

app.post("/api/admin/resetpw", adminAuth, (req, res) => {
  const { userId, newPassword } = req.body;
  try {
    const users = read("users.json");
    const user = users.find(u => u.id === userId);
    if (!user) return res.json({ success: false, message: "User not found" });
    const salt = crypto.randomBytes(16).toString("hex");
    user.password_salt = salt;
    user.password_hash = hashPassword(newPassword, salt);
    write("users.json", users);
    res.json({ success: true, message: "Password reset success" });
  } catch (error) {
    res.json({ success: false, message: "Server error" });
  }
});

app.post("/api/admin/toggle-suspend", adminAuth, (req, res) => {
  const { userId } = req.body;
  try {
    const users = read("users.json");
    const user = users.find(u => u.id === userId);
    if (!user) return res.json({ success: false });
    user.suspended = !user.suspended;
    write("users.json", users);
    res.json({ success: true });
  } catch (error) {
    res.json({ success: false });
  }
});

app.post("/api/admin/update-package", adminAuth, (req, res) => {
  const { userId, packageName } = req.body;
  try {
    const users = read("users.json");
    const packages = read("packages.json");
    const user = users.find(u => u.id === userId);
    const pack = packages.find(p => p.name === packageName);
    if (!user || !pack) return res.json({ success: false, message: "Not found" });

    let limitValue = pack.limit;
    if (limitValue === "unlimited") {
      limitValue = 999999;
    } else {
      limitValue = parseInt(limitValue);
      if (isNaN(limitValue)) limitValue = 0;
    }

    user.package = pack.name;
    user.limit = limitValue;
    user.usage = 0;
    const expiresAt = new Date();
    expiresAt.setMonth(expiresAt.getMonth() + (pack.duration || 1));
    user.expiresAt = expiresAt.toISOString();
    write("users.json", users);
    res.json({ success: true, message: "Package updated" });
  } catch (error) {
    res.json({ success: false, message: "Server error" });
  }
});

// ==================== ADMIN SETTINGS API ====================
app.get("/api/admin/settings", (req, res) => {
  try {
    const settings = read("settings.json");
    res.json(settings);
  } catch (error) {
    const defaultSettings = { title: "Sky Rest API", description: "Powerful REST API Service" };
    write("settings.json", defaultSettings);
    res.json(defaultSettings);
  }
});

app.put("/api/admin/settings", adminAuth, (req, res) => {
  const { title, description, whatsapp, creator, contact } = req.body;
  try {
    const settings = { title, description, whatsapp, creator, contact };
    write("settings.json", settings);
    res.json({ success: true, message: "Settings updated" });
  } catch (error) {
    res.json({ success: false, message: "Server error" });
  }
});

// ==================== PACKAGES API ====================
app.get("/api/packages", (req, res) => {
  try {
    res.json(read("packages.json"));
  } catch (error) {
    res.json([]);
  }
});

app.get("/api/admin/packages", (req, res) => {
  try {
    res.json(read("packages.json"));
  } catch (error) {
    res.json([]);
  }
});

app.post("/api/admin/packages", adminAuth, (req, res) => {
  const { name, limit, duration, price } = req.body;
  if (!name || !limit) {
    return res.json({ success: false, message: "Name and limit required" });
  }
  try {
    const packages = read("packages.json");
    if (packages.find(p => p.name === name)) {
      return res.json({ success: false, message: "Package already exists" });
    }
    const newPackage = {
      name,
      price: parseInt(price),
      limit: parseInt(limit),
      duration: duration ? parseInt(duration) : 1
    };
    packages.push(newPackage);
    write("packages.json", packages);
    res.json({ success: true, message: "Package added" });
  } catch (error) {
    res.json({ success: false, message: "Server error" });
  }
});

app.put("/api/admin/packages/:name", adminAuth, (req, res) => {
  const oldName = req.params.name;
  const { name, limit, duration, price } = req.body;
  try {
    const packages = read("packages.json");
    const index = packages.findIndex(p => p.name === oldName);
    if (index === -1) {
      return res.json({ success: false, message: "Package not found" });
    }
    if (name !== oldName && packages.find(p => p.name === name)) {
      return res.json({ success: false, message: "Package name already exists" });
    }
    packages[index] = {
      name: name || packages[index].name,
      limit: limit ? parseInt(limit) : packages[index].limit,
      price: price ? parseInt(price) : packages[index].price,
      duration: duration ? parseInt(duration) : packages[index].duration
    };
    write("packages.json", packages);
    res.json({ success: true, message: "Package updated" });
  } catch (error) {
    res.json({ success: false, message: "Server error" });
  }
});

app.delete("/api/admin/packages/:name", adminAuth, (req, res) => {
  const name = req.params.name;
  try {
    const packages = read("packages.json");
    const filtered = packages.filter(p => p.name !== name);
    if (filtered.length === packages.length) {
      return res.json({ success: false, message: "Package not found" });
    }
    write("packages.json", filtered);
    res.json({ success: true, message: "Package deleted" });
  } catch (error) {
    res.json({ success: false, message: "Server error" });
  }
});

// ==================== CRON JOBS ====================

// Reset usage setiap hari jam 00:00 (hanya untuk user yang belum expired)
cron.schedule('0 0 * * *', () => {
  console.log(chalk.cyan('Menjalankan cron: reset usage harian...'));
  try {
    const users = read("users.json");
    let updated = false;
    users.forEach(user => {
      // Cek apakah user sudah expired? Jika expired, jangan reset usage
      if (user.expiresAt && new Date(user.expiresAt) >= new Date()) {
        if (user.usage !== 0) {
          user.usage = 0;
          updated = true;
        }
      }
    });
    if (updated) {
      write("users.json", users);
      console.log(chalk.green('Usage berhasil direset untuk user aktif.'));
    } else {
      console.log(chalk.yellow('Tidak ada user yang perlu direset.'));
    }
  } catch (err) {
    console.error(chalk.red('Gagal menjalankan cron reset usage:'), err);
  }
});

// Cek expired key setiap jam (misal setiap jam 0 menit)
cron.schedule('0 * * * *', () => {
  console.log(chalk.cyan('Menjalankan cron: pengecekan API key expired...'));
  try {
    const users = read("users.json");
    const now = new Date();
    const expiredUsers = users.filter(u => u.expiresAt && new Date(u.expiresAt) < now);
    if (expiredUsers.length > 0) {
      console.log(chalk.yellow(`Ditemukan ${expiredUsers.length} API key expired.`));
      // Bisa ditambahkan aksi misalnya suspend otomatis, tapi tidak diminta
    } else {
      console.log(chalk.green('Semua API key masih berlaku.'));
    }
  } catch (err) {
    console.error(chalk.red('Gagal menjalankan cron cek expired:'), err);
  }
});

// ==================== START SERVER ====================
app.listen(PORT, () => {
  console.log(chalk.bgHex('#90EE90').hex('#333').bold(` Server is running on port ${PORT} `));
  console.log(chalk.cyan(`  User Frontend: http://localhost:${PORT}`));
  console.log(chalk.cyan(`  Admin Login: http://localhost:${PORT}/admin/login`));
  console.log(chalk.cyan(`  API Documentation: http://localhost:${PORT}/api`));
  console.log(chalk.cyan(`  Settings API: http://localhost:${PORT}/settings`));
});